
import java.awt.*;
import java.awt.image.MemoryImageSource;

/**

Television.java - Copyright 2003-2008 by Michael Kohn 

Part of the Jatari project.  This module falls under the GPL license.

1 2 1 2 The Naken Crew

Email: mike@mikekohn.net
  Web: http://www.mikekohn.net/

Horizontal Scan: 68 clocks + 160 clock = 228 clocks
Vertical Scan: 3 clocks + 37 clocks + 192 clocks + 30 clocks = 262 clocks
Total Resolution: 228x262
Visible Resolution: 160x192

This class creates a memory image source of 160x(192*2) - to deal with
interlacing - and strech draws it on the window.

Clock Speed: 3.58MHz  (CPU clock * 3)

*/

public class Television extends Frame
{
final int TV_WIDTH=228;
final int TV_HEIGHT=262;
MemoryImageSource image_source;
public int posx=0;
public int posy=0;
int[] colors;
Image tvImage=null;
int raster_ptr=0;
int width,height;
boolean okay=false;
int offsetY=0;
long startTime;
int frame_count;
int[] image_pixels;
int raster_color=0;
int plane=0;

  public Television(int width, int height)
  {
    super();

    this.setSize(width,height);
    this.setVisible(true);

    this.width=width;
    this.height=height;

    raster_ptr=0;

    // startTime=System.currentTimeMillis();
  }

  public void init()
  {
  Insets ins;

    ins = this.getInsets();
    this.setSize(width+ins.left+ins.right,height+ins.top+ins.bottom);
    offsetY=ins.top;

    setupColorTable();

    image_pixels=new int[160*262*2];
    image_source=new MemoryImageSource(160, 192*2, image_pixels, 0, 160);
    image_source.setAnimated(true);
    tvImage=createImage(image_source);
  }

  /** Define all colors. */

  private void setupColorTable()
  {
  int c,cx,y,r,i;
  int[][] yuv_table=
    {
      { 255, 128, 128 },     // Grey 0
      { 235,  54, 140 },     // Gold 1
      { 201,  81, 152 },     // Orange 2
      { 202,  97, 163 },     // Bright Orange 3
      { 201, 115, 164 },     // Pink 4
      { 199, 141, 154 },     // Purple 5
      { 195, 159, 159 },     // Purple-Blue 6
      { 188, 180, 252 },     // Blue 7
      { 144, 172, 113 },     // Blue 8
      { 195, 160, 105 },     // Light-Blue 9
      { 166, 153,  97 },     // Torquois 10 
      { 221, 122,  87 },     // Green-Blue 11
      { 223, 105,  99 },     // Green-Yellow 12
      { 226,  92, 109 },     // Green 13
      { 223,  90, 128 },     // Orange-Green 14
      { 180,  87, 147 }      // Light-Orange 15
    };

    colors=new int[128];
    c=0;

    while(c<128)
    {
      if (c==0)
      { y=0; }
        else
      { y=8; }

      i=c/8;
      cx=yuv_table[i][0]/8;
      for (r=0; r<8; r++)
      {
        colors[c++]=yuv2rgb(y,yuv_table[i][1],yuv_table[i][2]);
        y=y+cx;
      }
    }
  }

  /** Get the current raster beam X position (includes blanking area) */

  public int getX()
  {
    return posx;
  }

  /** Get the current raster beam Y position (includes blanking area) */

  public int getY()
  {
    return posy;
  }

  /** Apply 1 pixel clock.  Draw the current raster beam color in this area */

  public void clock()
  {
    if (posx>=68 && posy>40 && posy<232)
    {
      image_pixels[raster_ptr++]=raster_color;
    }

    posx++;
    if (posx>=TV_WIDTH)
    {
      // Hsync!
      posx=0;
      raster_ptr+=160; // interlacing
      posy++;
    }
  }

  /** Apply n pixel clocks.  Draw the current raster beam color in this area */

  public void clock(int n)
  {
  int c,blank;

    while(n>0)
    {
      c=TV_WIDTH-posx;
      if (c>n) c=n;
      n=n-c;
      if (posx<68)
      {
        blank=68-posx;
        if (blank<=c)
        {
          posx=posx+c;
          continue;
        }
        c=c-blank;
        posx=68;
      }

      if (posx>=68 && posy>40 && posy<232)
      {
        image_pixels[raster_ptr++]=raster_color;
      }

      posx=posx+c;
      if (posx>=TV_WIDTH)
      {
        // Hsync!
        posx=0;
        raster_ptr+=160; // interlacing
        posy++;
      }
    }
  }

  /** Apply vsync pulse to TV.  Usually this lasts 3 scan lines but
     we'll assume all is okay from a single pulse */

  public void vsync()
  {
    image_source.newPixels(0,0,160,192*2);
    repaint();
    //invalidate();
    posx=0;
    posy=0;

    frame_count++;
    plane++;

    if ((plane%2)==1) { raster_ptr=160; }
    else { raster_ptr=0; }

    if (System.currentTimeMillis()-startTime>=1000)
    {
      System.out.println("fps="+(float)(frame_count)/((float)(System.currentTimeMillis()-startTime)/1000));
      startTime=System.currentTimeMillis();
      frame_count=0;
    }
  }

  /** Change the raster beam color to a preset Atari color */

  public void setColor(int c)
  {
    raster_color=colors[c&0x7f];
  }

  /** Change the raster beam color to a YUV color */

  public void setColor(int y, int u, int v)
  {
    raster_color=yuv2rgb(y,u,v);
  }

  /** Paint the screen. Not needed to be called by other classes. */

  public void update(Graphics gg)
  {
    //gg.drawImage(tvImage,0,0,width,height,0,0,160,222*2,null);
    gg.drawImage(tvImage,0,0,width,height,0,0,160,192*2,null);
  }

  public void paint(Graphics gg)
  {
    //gg.drawImage(tvImage,0,0,width,height,0,0,160,222*2,null);
    gg.drawImage(tvImage,0,0,width,height,0,0,160,192*2,null);
  }

  /** Create a new Java color from y,u,v */

  int yuv2rgb(int y, int u, int v)
  {
  double r,g,b;

    u=u-128;
    v=v-128;

    r=(double)y+(double)(1.4075 * ((double)v));
    g=(double)y-(double)(0.3455 * ((double)u)) - (double)(0.7169*((double)v));
    b=(double)y+(double)(1.7790 * ((double)u));

    if (r<0) r=0;
    if (g<0) g=0;
    if (b<0) b=0;

    if (r>255) r=255;
    if (g>255) g=255;
    if (b>255) b=255;

    return (0xff<<24)|(((int)r)<<16)|((int)g<<8)|(int)b;
  }
}


