#!/usr/bin/perl

# perl_kohn_bmp - Copyright 2008 by Michael Kohn
# http://www.mikekohn.net/
# mike@mikekohn.net
#
# Feel free to use this class in your own programs (commerical or not)
# as long as you don't remove my name, email address, webpage, or copyright.
#
# Example how to use:
#
# my_bmp.kohn_bmp("out.bmp",image_width,image_height,3) <-- depth of 3 is color
# my_bmp.write_pixel(red,green,blue)  <-- do this width*height times
# my_bmp.close()
#
# if depth is set to 1 (black and white image) call write_pixel_bw(y) where
# y is between 0 and 255

package KohnBMP;

use IO::File;
use strict;
use warnings;

sub write_int
{
  my ($self,$n)=@_;

  $self->{out}->write(pack("V",$n));
}

sub write_word
{
  my ($self,$n)=@_;

  $self->{out}->write(pack("v",$n));
}

sub new
{
  my ($class,$filename,$width,$height,$depth)=@_;
  my $self=();

  $self->{width}=$width;
  $self->{height}=$height;
  $self->{depth}=$depth;
  $self->{width_bytes}=$width*$depth;

  $self->{out}=0;
  $self->{bytes}=0;
  $self->{xpos}=0;

  bless $self, $class;

  if (($self->{width_bytes}%4)!=0)
  {
    $self->{width_bytes}=$self->{width_bytes}+(4-($self->{width_bytes}%4));
  }

  $self->{out}=new IO::File;
  $self->{out}->open(">".$filename);
  $self->{out}->write("BM");   # BM stands for bowel movement

  if ($depth==1)
  {
    $self->write_int(($self->{width_bytes}*$height)+54+1024)
  }
    else:
  {
    $self->write_int(($self->{width_bytes}*$height)+54)
  }

  $self->write_word(0);
  $self->write_word(0);

  if ($depth==1)
  {
    $self->write_int(54+1024);
  }
    else:
  {
    $self->write_int(54);
  }

  $self->write_int(40);                # header_size
  $self->write_int($width);             # width
  $self->write_int($height);            # height
  $self->write_word(1);                # planes
  $self->write_word($depth*8);          # bits per pixel
  $self->write_int(0);                 # compression
  $self->write_int($self->{width_bytes}*$height*$depth); # image_size
  $self->write_int(0);                 # biXPelsperMetre
  $self->write_int(0);                 # biYPelsperMetre

  if ($depth==1)
  {
    $self->write_int(256);             # colors used
    $self->write_int(256);             # colors important

    for (my $c=0; $c<256; $c++)
    {
      $self->{out}->write(sprintf("%c",$c));
      $self->{out}->write(sprintf("%c",$c));
      $self->{out}->write(sprintf("%c",$c));
      $self->{out}->write(sprintf("%c",0));
    }
  }
    else
  {
    $self->write_int(0);               # colors used - 0 since 24 bit
    $self->write_int(0);               # colors important - 0 since 24 bit
  }

  return $self;
}

sub write_pixel_bw
{
  my ($self,$y)=@_;

  $self->{out}->write(sprintf("%c",$y));
  $self->{xpos}=$self->{xpos}+1;

  if ($self->{xpos}==$self->{width})
  {
    while ($self->{xpos}<$self->{width_bytes})
    {
      $self->{out}->write(sprintf("%c",0));
      $self->{xpos}=$self->{xpos}+1;
    }
    $self->{xpos}=0;
  }
}

sub write_pixel
{
  my ($self,$red,$green,$blue)=@_;

  $self->{out}->write(sprintf("%c",($blue&255)));
  $self->{out}->write(sprintf("%c",($green&255)));
  $self->{out}->write(sprintf("%c",($red&255)));

  $self->{xpos}=$self->{xpos}+1;
  if ($self->{xpos}==$self->{width})
  {
    $self->{xpos}=$self->{xpos}*3;

    while ($self->{xpos}<$self->{width_bytes})
    {
      $self->{out}->write(sprintf("%c",0));
      $self->{xpos}=$self->{xpos}+1;
    }

    $self->{xpos}=0;
  }
}

sub close
{
  my ($self)=@_;

  close($self->{out});
}

1; 

